﻿// Applicable license: Code Project Open License (CPOL) 1.02
using System;
using System.Collections;
using System.Collections.Generic;
using System.Data;
using System.IO;
using System.Text;
using System.ComponentModel;
using System.Threading;
using System.Drawing;
using System.Drawing.Imaging;

namespace ParadoxTest
{
    partial class Program
    {
        // jvd 20250113 /////////////////////////////////////////////////////////////////////////////
        //Mr. Mitchell: 
        //Blob fields in DB File:
        //- variable length
        //- uses an extra 10 bytes in the DB record
        //- when the entire Blob will fit in the leader, the blob is stored in the leader and is not written to the MB file.
        //- In a DB record, a blob field is stored as a fixed-length data field (called the leader) followed by 10 bytes with the following fields:
        //    • offset (32bits)
        //    • index value
        //    • blob length (ulong)
        //    • modification number from the MB file header ( unsigned short integer (16 bits))
        //- Binary fields have zero-length  leader
        //- Memo fields have a leader of at least 1 byte length.
        //- Numeric data in a DB record stored in modified big endian format
        //- The 10 bytes leader of blob information are stored in little endian
        //- MB_Offset: first four bytes after/of (?) the leader, locates the blob data in the MB file.
        //- MB_Offset = 0 then the entire blob is contained in the leader.
        //- MB_Index: is the low-order byte from MB_Offset - {Change the low-order byte of MB_Offset to zero.???}
        //- MB_Index = FFh, then MB_Offset contains the offset of a Type 02 block in the MB file.
        //-  MB_Index != Ffh, then
        //    • MB_Offset contains the offset of a type 03 block in the MB file
        //    • MB_Index contains the index of an entry in the Blob Pointer Array in the type 03 block.      
        /// //////////////////////////////////////////////////////////////////////////////////////////
        private static byte[] ProcessGraphicBlobType(byte[] S2_11byte, string mbPathFile, int recnr, string strFldName, out int iMBblockType)
        {
            int iEventCatched = 0;
            string tmp = BitConverter.ToString(S2_11byte); // jvd for logging
            tmp = tmp.Replace('-', ' ');
//eg:       42 FF 50 00 00 4A 08 00 00 01 00
            iMBblockType = -1;

            //- The 10 bytes leader of blob information are stored in little endian
            //- MB_Offset: first four bytes after/of (?) the leader, locates the blob data in the MB file.
            //- MB_Offset = 0 then the entire blob is contained in the leader.
            //- MB_Index: is the low-order byte from MB_Offset - {Change the low-order byte of MB_Offset to zero.???}
            //- MB_Index = FFh, then MB_Offset contains the offset of a Type 02 block in the MB file.

            byte[] bOffset = new byte[4];
            byte[] bBlob = null;

//          This works OK, and is the most compact approach!
            long lMB_Offset = S2_11byte[3] + 256 * S2_11byte[2];      // big endian
            byte bMB_Index = S2_11byte[1];
            if (bMB_Index == 0xFF)  // then MB_Offset contains the offset of a Type 02 block in the MB file.
            {
                //    PARADOX 4.x FILE FORMATS, Revision 1,  May 11, 1996
                //    Author: Kevin Mitchell
                //
                //Single Blob Block (Type 02)
                //---------------------------
                //A single blob block can appear anywhere in the MB file. A "long" blob is stored in this kind of block.
                //The block length is the smallest multiple of 4k that is greater than or equal to the length of the blob.
                //The block header contains the following fields.
                //Hex   Field
                //Offset Type  Description
                //------ ----- --------------------------------------------------
                //000000 UB    Record type = 02h (block contains one blob)
                //000001 US    Size of block divided by 4k
                //000003 UL    Length of the blob.
                //000007 US    Modification number. This is reset to 1 by a table restructure.
                //000009       Blob data starts here
                iMBblockType = 2;
                int iBlockLen = 4096;
                byte[] bytebuffer = ReadFromMBFile(mbPathFile, lMB_Offset, iBlockLen);
                bBlob = ProcessType2Block(bytebuffer, (byte[])S2_11byte);
            }
            else
            {
                iMBblockType = 3;
            }
            int iBlobLen = S2_11byte[5] + 256 * S2_11byte[6];         // little endian
            bBlob = ReadFromMBFile(mbPathFile, lMB_Offset, iBlobLen);

            // save graphic blob to jpg? file:
            string filePath = "Recnr_" + recnr.ToString() + "_" + strFldName + "_image.jpg";
            File.WriteAllBytes(filePath, bBlob); // Requires System.IO

            if (bBlob == null)
                iMBblockType = -1;
            return bBlob;
        }   // void ProcessGraphicBlobType(..)

        public static Image ByteArrayToImage(byte[] byteArray)
        {
            try
            {
                using (MemoryStream stream = new MemoryStream(byteArray))
                {
                    return Image.FromStream(stream, true, true);
                }
            }
            catch (Exception e)
            {
                string strError = e.Message;
            }
            return null;
        }

        private static byte[] ProcessBlobType(byte[] GraphicBLobArray, string mbPathFile, int recnr, string strFldName, out int iMBblockType)
        {
            string tmp = BitConverter.ToString(GraphicBLobArray); // jvd even tbv verslag
            tmp = tmp.Replace('-', ' ');
            //bv:       42 FF 50 00 00 4A 08 00 00 01 00

            iMBblockType = -1;
            int iTyp2ExtrsOffset = 0;
            long lMB_Offset = GraphicBLobArray[3] + 256 * GraphicBLobArray[2];      // big endian
            byte bMB_Index = GraphicBLobArray[1];
            if (bMB_Index == 0xFF)
            {
                iMBblockType = 2;
                iTyp2ExtrsOffset = 9;       // jvd 20250118 only for ALBUM.DB: found by try and error, not documented anywhere
                if (mbPathFile.ToLower().Contains("album") == true)
                {
                    // these codelines show how cripple Paradox can be!!
                    if (recnr < 3 && strFldName.Contains("nail") == true)
                        iTyp2ExtrsOffset = 17;      // jvd 20250118 only for ALBUM.DB: found by try and error, not documented anywhere
                    else
                        if (recnr == 5)
                            iTyp2ExtrsOffset = 16;      // jvd 20250118 only for ALBUM.DB: found by try and error, not documented anywhere
                        else
                            if (recnr == 7)
                                iTyp2ExtrsOffset = 14;      // jvd 20250118 only for ALBUM.DB: found by try and error, not documented anywhere
                            else
                                if (recnr == 9)
                                    iTyp2ExtrsOffset = 13;      // jvd 20250118 only for ALBUM.DB: found by try and error, not documented anywhere
                }
            }
            else
            {
                iMBblockType = 3;
//              Logger.log("Found Type3 in  ProcessBlobType(..)");
            }

            int iBlobLen = GraphicBLobArray[5] + 256 * GraphicBLobArray[6];         // little endian
            byte[] bBlob = null;
            bBlob = ReadFromMBFile(mbPathFile, lMB_Offset + iTyp2ExtrsOffset, iBlobLen);

            // save blob to jpg file:
            string strExt = ".jpg";

           //get the format/file type
           ImageFormat imf = ImageDetector.GetImageType(bBlob);
           if (imf != null)
           {
               strExt = "." + imf.ToString();

               string filePath = "Recnr_" + recnr.ToString() + "_" + strFldName + "_image" + strExt;
               File.WriteAllBytes(filePath, bBlob); // Requires System.IO
           }
 
            if (bBlob == null)
                iMBblockType = -1;
            return bBlob;
        }   // void ProcessBlobType(..)


        // jvd 20241201
        private static byte[] ProcessType2Block(byte[] bytebuffer /*OneKiloByteBlock*/, byte[] S2_11byte)
        {
//          string strMemoText = string.Empty;

            //    PARADOX 4.x FILE FORMATS, Revision 1,  May 11, 1996
            //    Author: Kevin Mitchell
            //
            //Single Blob Block (Type 02)
            //---------------------------
            //A single blob block can appear anywhere in the MB file. A "long" blob is stored in this kind of block.
            //The block length is the smallest multiple of 4k that is greater than or equal to the length of the blob.
            //The block header contains the following fields.
            //Hex   Field
            //Offset Type  Description
            //------ ----- --------------------------------------------------
            //000000 UB    Record type = 02h (block contains one blob)
            //000001 US    Size of block divided by 4k
            //000003 UL    Length of the blob.
            //000007 US    Modification number. This is reset to 1 by a table restructure.
            //000009       Blob data starts here

            string tmp = BitConverter.ToString(S2_11byte);
            tmp = tmp.Replace('-', ' ');    // tmp = 3C FF 40 00 00 22 0E 00 00 01 00

            int iRecrdType = bytebuffer[0];

            // alternative method: 
            long lOffset1 = S2_11byte[3] + 256 * S2_11byte[2];      // big endian:      64*256 = 16384d
            int iMemoLen = S2_11byte[5] + 256 * S2_11byte[6];       // little endian:   34 + 14*256 = 3618d 

            // method according the above description of a Single Blob Block (Type 02):
            int iBlockSize = bytebuffer[1] * 4096;                  // 1*4096
            byte[] bLen = new byte[4];
            Array.Copy(bytebuffer, 3, bLen, 0, 4);                  // 0 0 14 34    34 + 14*256 + 0 + 0 = 3618
//          Array.Reverse(bLen);
            long lBlockLen = BitConverter.ToInt32(bLen, 0);         // 3618     OK, I think this is an Int32 in stead of a Int64 (Long)
            int iBlobStart = 9;                                     // see on 4009h

            byte[] memoBytes = new byte[lBlockLen];

            if ( memoBytes.Length <= bytebuffer.Length)
                Array.Copy(bytebuffer, 9, memoBytes, 0, memoBytes.Length);

            return memoBytes;

        }   // byte[] ProcessType2Block(..)


        // jvd 20241201
        private static byte[] ProcessType3Block(byte[] ByteBlock, byte[] S2_11byte, int iPntrArrayBlokNr)
        {
//          string strMemoText = string.Empty;
            byte[] bBlob = null;

            V3PntrArray v3MBPtrArrays = VerzamelV3PntrArrays(ByteBlock, S2_11byte);

            if (v3MBPtrArrays != null && v3MBPtrArrays.getArraysCount() > 0)
            {
                byte[] ptrArray = v3MBPtrArrays.getPtrArray(iPntrArrayBlokNr);
//              string tmp = BitConverter.ToString(ptrArray);   // 21-48-01-00-16 hex: 3F 15 30 01 00 10

                if (ptrArray != null)
                {
//                 string strPointerArray = BitConverter.ToString(ptrArray);
                    bBlob = HaalBijBehorendeGegevensUitOneKiloByteBlock(S2_11byte, ByteBlock, ptrArray);
                }
                else
                    Logger.log("BlokNr " + iPntrArrayBlokNr.ToString() + "S2_11byte " + "unable to retrieve PointerArray and memoBlob tekst");
            }

            return bBlob;
        }   // byte[] ProcessType3Block(..)



        // jvd 20241123
        //      For example, if an array entry looks like: 25-03-0F-00-07 then the data associated with the entry
        //      starts at offset 0250h (25h times 10h) and has 10h times 03h bytes allocated (48 bytes).
        // ???  The actual data length is 27h (39 bytes) because there are only 7 bytes of data in the last 16 byte chunk.
        //      The modification number is in little endian format and is 000Fh (15).
        private static byte[] HaalBijBehorendeGegevensUitOneKiloByteBlock(byte[] ElevenDBPtrArray, byte[] OneKiloByteBlock, byte[] PntrArray)
        {
//          string strRet = string.Empty;

            if (OneKiloByteBlock.Length == 0)
                return null;

            int iBlokNr = ElevenDBPtrArray[1];
            long lOffset1 = ElevenDBPtrArray[3] + 256 * ElevenDBPtrArray[2];   // big endian
            int iMemoLen = ElevenDBPtrArray[5] + 256 * ElevenDBPtrArray[6];    // little endian

            int iOffset2 = 16 * PntrArray[1];
            int iLen = 16 * PntrArray[2];           // unknown use ?????

            byte[] bytebuffer = new byte[iMemoLen];
            Array.Copy(OneKiloByteBlock, iOffset2, bytebuffer, 0, bytebuffer.Length);

//          strRet = ConvertAndFilterBytesToChars(bytebuffer, bytebuffer.Length);

            return bytebuffer;
        }   // byte[] HaalBijBehorendeGegevensUitOneKiloByteBlock(..)



        // jvd 20241113 - search for pointer arrays in mb file:
        private static V3PntrArray VerzamelV3PntrArrays(byte[] OneKiloByteBlock, byte[] MemoBLobArray)
        {
            V3PntrArray myV3PntrArray = null;

            int iPntrArrayBlokNr = MemoBLobArray[1], iEventCatched = 0, iType = 0, iT0Cnt = 0, iT1Cnt = 0, iT2Cnt = 0;
            long lOffset1 = MemoBLobArray[3] + 256 * MemoBLobArray[2];  // big endian
            int iMemoLen = MemoBLobArray[5] + 256 * MemoBLobArray[6];   // little endian

            int iRawBlokLen = 0x1150 - 0x1010 + 0x10;
            byte[] bytebuffer = new byte[iRawBlokLen];

            //// read whole Pointer area of 320 bytes excluding first 16 bytes which hold the block type:
            Array.Clear(bytebuffer, 0, bytebuffer.Length);
            Array.Copy(OneKiloByteBlock, 0, bytebuffer, 0, iRawBlokLen);
            iEventCatched = 0;

            //////////////////////////////////////////////////////////////////////////
            //          Block types:
            //          00 - Header block
            //          02 - Single blob block
            //          03 - Suballocated block
            //          04 - Free block
            //////////////////////////////////////////////////////////////////////////
            // is it a v2 or a v3 type?
            iType = bytebuffer[0];
            if (iType == 3)
            {
                ///////////////////////////////////////////////////////////////////////////////////////////////////////
                //    PARADOX 4.x FILE FORMATS, Revision 1,  May 11, 1996
                //    Author: Kevin Mitchell
                //
                //  Suballocated Block (Type 03)
                //  ----------------------------
                //  A suballocated block can appear anywhere in the MB file.
                //  Up to 64 short blobs may be stored in this type of block.
                //  A suballocated block is 4k bytes long. It has a 12-byte header followed by an array of up to 64 5-byte blob pointers.
                //  The DB field that "owns" a blob contains the offset of the block (from the start of the MB file) and the index of one
                //  of the entries in the blob pointer array. The array entry points to the blob data.
                //
                //  This method of using indirect pointers (a pointer to a pointer) is quite common. It simplifies garbage collection.
                //  Paradox can move data around within the block to consolidate non-contiguous chunks of free space.
                //  All it has to do is update the pointer array within the block.

                //  The 12-byte block header contains the following fields.
                //   Hex   Field
                //  Offset Type  Description
                //  ------ ----- --------------------------------------------------
                //  000000 UB    Record type = 03h (Suballocated block)
                //  000001 US    Size of block divided by 4k
                //               1 because the block size is 4k
                //  Note: There are nine more bytes in the header. I have no idea what they contain.
                //
                //  The blob pointer array follows the header. The array has 64 entries numbered from 00h to 3Fh. 
                //  Entries are used in reverse order. The 3Fh entry is used first. Then the 3Eh entry, then 3D, and so on ..
                //  The offset (from the start of the block) of the entry indexed by i is calculated as: offset = 12 + ( 5 * i )
                //  Each entry is 5 bytes long and has the following format.
                //  Hex          Field
                //  Offset Type  Description
                //  000000 UB    Data offset divided by 16
                //  000000 UB    Data offset divided by 16
                //               The offset is measured from start of the 4k block. If this is zero, then the blob was deleted and
                //               the space has been reused for another blob (which is associated with another entry in the array).
                //  000001 UB    Data length divided by 16 (rounded up)
                //  000002 US    Modification number from blob header. This is reset to 1 by a table restructure.
                //  000004 UB    Data length modulo 16. If this is zero, then the associated blob has been deleted and the space
                //               can be reused for an active blob, this value will be between 01h and 10h

                //  Note: Suballocations are made in 16-byte chunks. The first available chunk is at offset 0150h in the block.
                //  Multiply the first byte of the pointer array entry by 16 to get the offset. The next byte is the number of chunks.
                //  The last byte tells you how many bytes of data there are in the last chunk.
                //  I don't know the purpose of the modification number.
                //
                //  For example, if an array entry looks like: 25030F0007 then the data associated with the entry starts 
                //  at offset 0250h (25h times 10h) and has 10h times 03h bytes allocated (48 bytes).
                //  The actual data length is 27h (39 bytes) because there are only 7 bytes of data in the last 16 byte chunk.
                //  The modification number is in little endian format and is 000Fh (15).
                ///////////////////////////////////////////////////////////////////////////////////////////////////////

                // pass first 0x10 bytes and all zero bytes there after:
                int ix = 0x10;

                //// calculate the number of V3PntrArray's to be found in the area:
                int iPntArrayCnt = (iRawBlokLen - ix) / 5;
                byte[] PntrArrayBuffer = new byte[iRawBlokLen - ix];
                Array.Copy(bytebuffer, ix, PntrArrayBuffer, 0, PntrArrayBuffer.Length);

                // this class extracts all V3PntrArray's in current 320bytes area!
                myV3PntrArray = new V3PntrArray(iPntrArrayBlokNr, iType, PntrArrayBuffer);
            }

            return myV3PntrArray;
        }   // private List<long> VerzamelV3PntrArrays(..)

        // jvd 20241114
        public class V3PntrArray
        {
            private int iBlokNr = 0, iArrayCnt = 0, iBlokType = 0, iPntrBlockCnt = 0;
            private byte[] FiveBytePntrBlok = null;     //new byte[5];
            private List<byte[]> lstPntrBlokken = new List<byte[]>();

            // constructor:
            public V3PntrArray(int iBnr, int iType, byte[] PntrZone)
            {
                iBlokNr = iBnr;
                iBlokType = iType;
                Extract5BytePointers(PntrZone);
            }

            // jvd 20241114
            private void Extract5BytePointers(byte[] PntrZone)
            {
                int ix = 0, iPtrNr = 0;
                while (ix < PntrZone.Length && PntrZone[ix] == 0x0)
                    ix++;

                // calculate number of FivePointerBlocks to facilitate reverse zero-based numbering of PointerBlocks
                iPntrBlockCnt = (PntrZone.Length - ix) / 5;     // PntrZone.Length moet hier 320 zijn zijn!
                iPtrNr = PntrZone.Length / 5 - iPntrBlockCnt;
                int iExtraEntry = 1;
                while (ix < PntrZone.Length - 5)
                {
                    FiveBytePntrBlok = new byte[5 + iExtraEntry];     // + 1 voor extra array entry to hold an array index
                    Array.Clear(FiveBytePntrBlok, 0, 5 + iExtraEntry);
                    FiveBytePntrBlok[0] = (byte)iPtrNr;
                    Array.Copy(PntrZone, ix, FiveBytePntrBlok, iExtraEntry, 5);
                    lstPntrBlokken.Add(FiveBytePntrBlok);
                    iArrayCnt++;
                    iPtrNr++;
                    ix += 5;
                }
            }   // void Extract5BytePointers(byte[] PntrZone)

            // jvd 20241114
            public byte[] getPtrArray(int iNr)
            {
                byte[] byteRet = null;
                int ix = 0;
                for (ix = 0; ix < lstPntrBlokken.Count; ix++)
                {
                    if (lstPntrBlokken[ix][0] == iNr)
                    {
                        byteRet = lstPntrBlokken[ix];
                        break;
                    }
                }
                return byteRet;
            }   // byte[] getPtrArray(..)

            // jvd 20241114
            public int getArraysCount()
            {
                return iArrayCnt;
            }

            // jvd 20241114
            public int getBlokNr()
            {
                return iBlokNr;
            }

            // jvd 20241114
            public int getBlokType()
            {
                return iBlokType;
            }
        }   // public class V3PntrArray

        // jvd 20250113
        private static byte[] ReadFromMBFile(string strMBTablePathName, long lOffset1, int iBlobLen)
        {
            if (File.Exists(strMBTablePathName) == false || lOffset1 < 0L)
                return null;
            FileStream fileStream = new FileStream(strMBTablePathName, FileMode.Open);
            fileStream.Seek(lOffset1, SeekOrigin.Begin);

            byte[] bytebuffer = new byte[iBlobLen];
            int iGelezen = fileStream.Read(bytebuffer, 0, bytebuffer.Length);
            fileStream.Close();
            if (iGelezen == bytebuffer.Length)
                return bytebuffer;
            else
                return null;
        }   // byte[] ReadFromMBFile(..)


 
    }   // public partial class Form1 : Form

}   // namespace ParadoxReader